/**
 * CLX GMaps AI Review - Website Content Script
 * Automatically gets UUID from extension and stores in localStorage
 */

(function() {
  'use strict';

  // Only run on our website
  const allowedHostnames = ['clx-gmaps-ai-review.celox.io', 'mapsmate.de', 'www.mapsmate.de'];
  if (!allowedHostnames.some(host => window.location.hostname.includes(host))) {
    return;
  }

  console.log('[CLX Website] Content script loaded');

  /**
   * Get UUID from extension storage
   */
  async function getUUIDFromExtension() {
    return new Promise((resolve) => {
      if (!chrome || !chrome.runtime || !chrome.runtime.sendMessage) {
        console.log('[CLX Website] Chrome runtime not available');
        resolve(null);
        return;
      }

      try {
        chrome.runtime.sendMessage({ action: 'getUUID' }, (response) => {
          if (chrome.runtime.lastError) {
            console.log('[CLX Website] Extension not installed or not responding:', chrome.runtime.lastError.message);
            resolve(null);
            return;
          }
          
          if (!response) {
            console.log('[CLX Website] No response from extension');
            resolve(null);
            return;
          }
          
          const uuid = response?.uuid || null;
          console.log('[CLX Website] UUID from extension:', uuid);
          resolve(uuid);
        });
      } catch (error) {
        console.error('[CLX Website] Error sending message to extension:', error);
        resolve(null);
      }
    });
  }

  /**
   * Store UUID in localStorage and update URL if needed
   */
  async function syncUUID() {
    const uuid = await getUUIDFromExtension();
    
    if (uuid) {
      // Store in localStorage
      localStorage.setItem('clx_gmaps_uuid', uuid);
      console.log('[CLX Website] UUID stored in localStorage:', uuid);
      
      // Update URL if UUID not present
      const urlParams = new URLSearchParams(window.location.search);
      const hashParams = new URLSearchParams(window.location.hash.substring(window.location.hash.indexOf('?') + 1));
      const urlUuid = urlParams.get('uuid') || hashParams.get('uuid');
      
      if (!urlUuid) {
        // Add UUID to URL
        const newUrl = new URL(window.location);
        if (newUrl.hash.includes('#')) {
          newUrl.hash = newUrl.hash.split('?')[0] + '?uuid=' + uuid;
        } else {
          newUrl.searchParams.set('uuid', uuid);
        }
        window.history.replaceState({}, '', newUrl);
        console.log('[CLX Website] UUID added to URL');
      }
    } else {
      // Try to get from localStorage as fallback
      const storedUuid = localStorage.getItem('clx_gmaps_uuid');
      if (storedUuid) {
        console.log('[CLX Website] Using UUID from localStorage:', storedUuid);
        const urlParams = new URLSearchParams(window.location.search);
        const hashParams = new URLSearchParams(window.location.hash.substring(window.location.hash.indexOf('?') + 1));
        const urlUuid = urlParams.get('uuid') || hashParams.get('uuid');
        
        if (!urlUuid) {
          const newUrl = new URL(window.location);
          if (newUrl.hash.includes('#')) {
            newUrl.hash = newUrl.hash.split('?')[0] + '?uuid=' + storedUuid;
          } else {
            newUrl.searchParams.set('uuid', storedUuid);
          }
          window.history.replaceState({}, '', newUrl);
        }
      }
    }
  }

  // Sync UUID when page loads
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', syncUUID);
  } else {
    syncUUID();
  }

  // Also sync when page becomes visible (in case extension was installed after page load)
  document.addEventListener('visibilitychange', () => {
    if (!document.hidden) {
      syncUUID();
    }
  });

})();

