/**
 * CLX GMaps AI Review - Content Script
 * DOM manipulation for Google Maps review response interface
 */

(function() {
  'use strict';

  console.log('[CLX Content] Script loaded on:', window.location.href);

  // Tonalities configuration
  const TONALITIES = {
    positive: [
      { id: 'herzlich', name: 'Herzlich' },
      { id: 'dankbar', name: 'Dankbar' },
      { id: 'enthusiastisch', name: 'Enthusiastisch' },
      { id: 'professionell', name: 'Professionell' },
      { id: 'humorvoll', name: 'Humorvoll' }
    ],
    neutral: [
      { id: 'verstaendnisvoll', name: 'Verständnisvoll' },
      { id: 'professionell', name: 'Professionell' },
      { id: 'einladend', name: 'Einladend' },
      { id: 'loesungsorientiert', name: 'Lösungsorientiert' },
      { id: 'freundlich', name: 'Freundlich' }
    ],
    negative: [
      { id: 'empathisch', name: 'Empathisch' },
      { id: 'entschuldigend', name: 'Entschuldigend' },
      { id: 'loesungsorientiert', name: 'Lösungsorientiert' },
      { id: 'witzig', name: 'Witzig' },
      { id: 'frech', name: 'Frech' }
    ]
  };

  const LENGTHS = [
    { id: 'kurz', name: 'Kurz' },
    { id: 'normal', name: 'Normal' },
    { id: 'lang', name: 'Lang' }
  ];

  let observer = null;

  /**
   * Initialize
   */
  function init() {
    console.log('[CLX Content] Initializing...');
    setupFrameListener(); // Setup cross-frame communication
    setupObserver();
    scanForReviews();
  }

  /**
   * Setup MutationObserver
   */
  function setupObserver() {
    if (observer) observer.disconnect();

    observer = new MutationObserver((mutations) => {
      clearTimeout(window.clxScanTimeout);
      window.clxScanTimeout = setTimeout(scanForReviews, 500);
    });

    observer.observe(document.body, { childList: true, subtree: true });
    console.log('[CLX Content] Observer active');
  }

  /**
   * Scan for review containers
   */
  function scanForReviews() {
    // Find all textareas for reviews
    const textareas = document.querySelectorAll('textarea[jsname="YPqjbf"]:not([data-clx-processed])');
    
    console.log(`[CLX Content] Found ${textareas.length} textareas`);
    
    textareas.forEach((textarea, index) => {
      // Skip if already processed
      if (textarea.dataset.clxProcessed) return;
      
      // Skip if textarea is not visible
      if (!textarea.offsetParent) {
        console.log(`[CLX Content] Textarea ${index} not visible`);
        return;
      }
      
      textarea.dataset.clxProcessed = 'true';
      
      // Find the review article - go up to find article.VaHEVc
      let article = textarea.closest('article.VaHEVc');
      
      // If not found, try going up through parent elements
      if (!article) {
        let parent = textarea.parentElement;
        let depth = 0;
        while (parent && depth < 10) {
          if (parent.classList.contains('VaHEVc') || parent.tagName === 'ARTICLE') {
            article = parent;
            break;
          }
          parent = parent.parentElement;
          depth++;
        }
      }
      
      // Find the button container - should be in .UP87Yb .DYNnmc .FkJOzc
      const buttonContainer = textarea.closest('.UP87Yb')?.querySelector('.DYNnmc .FkJOzc');
      
      if (buttonContainer) {
        console.log(`[CLX Content] Found button container for textarea ${index}`, {
          hasArticle: !!article,
          isInIframe: window !== window.top
        });
        // Add button even if article is not found (can happen in iframe)
        addCaptureButton(buttonContainer, textarea, article);
      } else {
        console.log(`[CLX Content] Could not find button container for textarea ${index}`);
      }
    });

    const count = document.querySelectorAll('.clx-capture-btn').length;
    if (count > 0) {
      console.log(`[CLX Content] ${count} Erfassen-Buttons aktiv`);
    }
  }

  /**
   * Add capture button
   */
  function addCaptureButton(buttonContainer, textarea, reviewArticle) {
    if (!buttonContainer) {
      console.log('[CLX Content] Button container not provided');
      return;
    }

    // Check if button already exists
    if (buttonContainer.querySelector('.clx-capture-btn')) {
      return;
    }

    // Create button
    const btn = document.createElement('button');
    btn.className = 'clx-capture-btn';
    btn.type = 'button';
    btn.innerHTML = `
      <svg viewBox="0 0 24 24" fill="currentColor" width="18" height="18">
        <path d="M12 2L14.39 8.26L21 9.27L16.5 13.97L17.77 20.72L12 17.27L6.23 20.72L7.5 13.97L3 9.27L9.61 8.26L12 2Z"/>
      </svg>
      <span>Erfassen</span>
    `;

    // Insert BEFORE the "Antworten" button (jsname="hrGhad")
    const antwortenBtn = buttonContainer.querySelector('button[jsname="hrGhad"]');
    if (antwortenBtn) {
      buttonContainer.insertBefore(btn, antwortenBtn);
      console.log('[CLX Content] Erfassen-Button vor "Antworten" eingefügt');
    } else {
      // Fallback: prepend to container
      buttonContainer.insertBefore(btn, buttonContainer.firstChild);
      console.log('[CLX Content] Erfassen-Button am Anfang eingefügt (Fallback)');
    }
    
    // Ensure button is visible
    btn.style.display = 'inline-flex';
    btn.style.visibility = 'visible';
    
    console.log('[CLX Content] Erfassen-Button hinzugefügt, Container:', buttonContainer.className);

    // Click handler
    btn.addEventListener('click', (e) => {
      e.preventDefault();
      e.stopPropagation();
      handleCapture(btn, textarea, reviewArticle || buttonContainer.closest('article.VaHEVc') || buttonContainer);
    });
  }

  /**
   * Handle capture click
   */
  async function handleCapture(btn, textarea, reviewContainer) {
    console.log('[CLX Content] Erfassen gestartet');

    btn.disabled = true;
    btn.innerHTML = '<span class="clx-spinner"></span><span>Analysiere...</span>';

    try {
      // Find the review article in the same DOM tree
      // The textarea is often in: .kx2i0d > article + .UP87Yb > textarea
      // Go up from textarea to find the container with the article
      let container = textarea.closest('.kx2i0d');
      if (!container) {
        // Fallback: go up until we find a container with an article
        container = textarea.parentElement;
        for (let i = 0; i < 20 && container; i++) {
          const articles = container.querySelectorAll('article');
          if (articles.length > 0) {
            console.log(`[CLX Content] Found ${articles.length} articles at parent level ${i}`);
            break;
          }
          container = container.parentElement;
        }
      }

      console.log('[CLX Content] Using container:', container?.className || 'not found');
      
      // Extract review data from the found container
      const reviewData = container ? extractReviewData(container) : { name: 'Kunde', text: '', stars: null };
      
      console.log('[CLX Content] Review data:', reviewData);

      // Check license
      const licenseResult = await sendMessage({ action: 'checkLicense' });
      if (!licenseResult.valid) {
        showError(btn, 'Lizenz abgelaufen. Bitte Plugin-Einstellungen prüfen.');
        resetButton(btn);
        return;
      }

      // Classify review
      const classification = await sendMessage({
        action: 'classifyReview',
        reviewText: reviewData.text,
        reviewerName: reviewData.name,
        starRating: reviewData.stars
      });

      console.log('[CLX Content] Classification:', classification);

      // Check classification result
      if (!classification || classification.error) {
        throw new Error(classification?.error || 'Klassifizierung fehlgeschlagen');
      }

      // Show panel
      showPanel(btn, textarea, reviewData, classification.type || 'neutral');

    } catch (error) {
      console.error('[CLX Content] Error:', error);
      const errorMessage = error.message || 'Ein Fehler ist aufgetreten';
      showError(btn, errorMessage);
    }

    resetButton(btn);
  }

  /**
   * Extract review data from container
   */
  function extractReviewData(container) {
    const data = { name: 'Kunde', text: '', stars: null };

    // If no container provided, we're in an iframe - can't extract review data
    if (!container) {
      console.log('[CLX Content] No container - running in iframe, cannot extract full review data');
      return data;
    }

    // Find the article element - try specific class first, then any article
    let article = container.classList?.contains('VaHEVc') ? container : 
                  container.querySelector('article.VaHEVc') || 
                  container.querySelector('article') || 
                  container;
    
    console.log('[CLX Content] Extracting from article:', article.tagName, article.className.substring(0, 50));

    // Find reviewer name: .PskQHd.jEgW2b or a[jsname="xs1xe"]
    const nameEl = article.querySelector('.PskQHd.jEgW2b, a[jsname="xs1xe"], .N0c6q a.PskQHd');
    if (nameEl) {
      data.name = nameEl.textContent.trim();
      console.log('[CLX Content] Found reviewer name:', data.name);
    }

    // Find star rating: .DYizzd[aria-label] contains "X von 5"
    const starsEl = article.querySelector('.DYizzd[aria-label], span[aria-label*="Stern"]');
    if (starsEl) {
      const ariaLabel = starsEl.getAttribute('aria-label') || '';
      const match = ariaLabel.match(/(\d)/);
      if (match) {
        data.stars = parseInt(match[1]);
        console.log('[CLX Content] Found stars from aria-label:', data.stars);
      }
    }
    
    // Count star icons as fallback - look for .lMAmUc icons within .DYizzd
    if (!data.stars) {
      const starIcons = article.querySelectorAll('.DYizzd i.lMAmUc, .DYizzd .material-icons-extended.lMAmUc');
      if (starIcons.length > 0) {
        data.stars = starIcons.length;
        console.log('[CLX Content] Found stars by counting icons:', data.stars);
      }
    }

    // Find review text: .gyKkFe.Fv38Af (the review text div) - must be substantial
    const textEls = article.querySelectorAll('.gyKkFe.Fv38Af, .gyKkFe.JhRJje.Fv38Af');
    for (const textEl of textEls) {
      const text = textEl.textContent.trim();
      // Must be substantial text, not metadata
      if (text.length > 20 && 
          !text.includes('Stern') && 
          !text.includes('Vor ') && 
          !text.includes('Local Guide') &&
          !text.includes('Rezension') &&
          !text.includes('Fotos') &&
          !text.match(/^\d+\.\d+\.\d{4}$/) && // Not a date
          !text.match(/^\d+\s+Rezension/) && // Not "X Rezensionen"
          !text.match(/^\d+\s+Fotos/)) { // Not "X Fotos"
        data.text = text;
        console.log('[CLX Content] Found review text:', data.text.substring(0, 50) + '...');
        break;
      }
    }

    // Final fallback: look for any substantial text in spans/divs
    if (!data.text || data.text.length < 10) {
      const allTexts = article.querySelectorAll('span, div');
      for (const el of allTexts) {
        const text = el.textContent.trim();
        // Must be substantial text, not metadata
        if (text.length > 20 && 
            !text.includes('Stern') && 
            !text.includes('Vor ') && 
            !text.includes('Local Guide') &&
            !text.includes('Rezension') &&
            !text.includes('Fotos') &&
            !text.match(/^\d+\.\d+\.\d{4}$/) &&
            !text.match(/^\d+\s+Rezension/) &&
            !text.match(/^\d+\s+Fotos/) &&
            !text.includes('Neu') &&
            !text.includes('Inhaber')) {
          data.text = text;
          console.log('[CLX Content] Found review text (fallback):', data.text.substring(0, 50) + '...');
          break;
        }
      }
    }

    console.log('[CLX Content] Extracted review data:', data);
    return data;
  }

  /**
   * Show generation panel
   */
  function showPanel(btn, textarea, reviewData, reviewType) {
    // Remove existing panel
    const existingPanel = document.querySelector('.clx-panel');
    if (existingPanel) existingPanel.remove();

    const typeLabels = { positive: 'Positiv', neutral: 'Neutral', negative: 'Negativ' };
    const stars = reviewData.stars ? '⭐'.repeat(reviewData.stars) : '';

    const panel = document.createElement('div');
    panel.className = 'clx-panel';
    panel.innerHTML = `
      <div class="clx-panel-header">
        <span class="clx-panel-title">Antwort generieren</span>
        <button class="clx-panel-close" type="button">✕</button>
      </div>
      <div class="clx-review-type ${reviewType}">
        ${stars} ${typeLabels[reviewType]}
      </div>
      <div class="clx-form-group">
        <label class="clx-form-label">Tonalität</label>
        <select class="clx-select clx-tonality-select">
          ${TONALITIES[reviewType].map((t, i) => 
            `<option value="${t.id}" ${i === 0 ? 'selected' : ''}>${t.name}</option>`
          ).join('')}
        </select>
      </div>
      <div class="clx-form-group">
        <label class="clx-form-label">Länge</label>
        <select class="clx-select clx-length-select">
          ${LENGTHS.map((l, i) => 
            `<option value="${l.id}" ${i === 1 ? 'selected' : ''}>${l.name}</option>`
          ).join('')}
        </select>
      </div>
      <div class="clx-form-group">
        <label class="clx-form-label">Anrede</label>
        <div class="clx-segmented-button">
          <button class="clx-segment" data-value="du" id="clx-segment-du">
            <span>Du</span>
          </button>
          <button class="clx-segment active" data-value="sie" id="clx-segment-sie">
            <span>Sie</span>
          </button>
        </div>
      </div>
      <button class="clx-generate-btn" type="button">
        <svg viewBox="0 0 24 24" fill="currentColor" width="18" height="18">
          <path d="M12 2L14.39 8.26L21 9.27L16.5 13.97L17.77 20.72L12 17.27L6.23 20.72L7.5 13.97L3 9.27L9.61 8.26L12 2Z"/>
        </svg>
        Generieren
      </button>
      <div class="clx-message"></div>
    `;

    // Insert panel after button
    btn.parentElement.insertBefore(panel, btn.nextSibling);

    // Load settings and update UI
    (async () => {
      try {
        const settingsResult = await sendMessage({ action: 'getSettings' });
        const formality = settingsResult?.formality || 'sie';
        updatePanelFormality(panel, formality);
        
        // Load last tonality for this review type
        const lastTonality = settingsResult?.lastTonality?.[reviewType];
        if (lastTonality) {
          const tonalitySelect = panel.querySelector('.clx-tonality-select');
          if (tonalitySelect) {
            tonalitySelect.value = lastTonality;
          }
        }
        
        // Load last length
        const lastLength = settingsResult?.lastLength || 'normal';
        const lengthSelect = panel.querySelector('.clx-length-select');
        if (lengthSelect) {
          lengthSelect.value = lastLength;
        }
      } catch (error) {
        console.error('[CLX Content] Failed to load settings:', error);
        // Default to 'sie'
        updatePanelFormality(panel, 'sie');
      }
    })();

    // Formality toggle handlers
    const duBtn = panel.querySelector('#clx-segment-du');
    const sieBtn = panel.querySelector('#clx-segment-sie');
    
    if (duBtn) {
      duBtn.addEventListener('click', () => updatePanelFormality(panel, 'du'));
    }
    if (sieBtn) {
      sieBtn.addEventListener('click', () => updatePanelFormality(panel, 'sie'));
    }

    // Close handler
    panel.querySelector('.clx-panel-close').addEventListener('click', () => panel.remove());

    // Generate handler
    panel.querySelector('.clx-generate-btn').addEventListener('click', async () => {
      const tonality = panel.querySelector('.clx-tonality-select').value;
      const length = panel.querySelector('.clx-length-select').value;
      const formality = getPanelFormality(panel);
      await handleGenerate(panel, textarea, reviewData, reviewType, tonality, length, formality);
    });
  }

  /**
   * Update formality selection in panel
   */
  function updatePanelFormality(panel, formality) {
    const duBtn = panel.querySelector('#clx-segment-du');
    const sieBtn = panel.querySelector('#clx-segment-sie');
    
    if (duBtn) {
      duBtn.classList.toggle('active', formality === 'du');
    }
    if (sieBtn) {
      sieBtn.classList.toggle('active', formality === 'sie');
    }
  }

  /**
   * Get selected formality from panel
   */
  function getPanelFormality(panel) {
    const duBtn = panel.querySelector('#clx-segment-du');
    return duBtn?.classList.contains('active') ? 'du' : 'sie';
  }

  /**
   * Handle generate click
   */
  async function handleGenerate(panel, textarea, reviewData, reviewType, tonality, length, formality) {
    const generateBtn = panel.querySelector('.clx-generate-btn');
    const messageEl = panel.querySelector('.clx-message');

    generateBtn.disabled = true;
    generateBtn.innerHTML = '<span class="clx-spinner"></span> Generiere...';
    messageEl.textContent = '';
    messageEl.className = 'clx-message';

    try {
      console.log('[CLX Content] Generating response...', { tonality, length, formality });
      
      const result = await sendMessage({
        action: 'generateResponse',
        reviewText: reviewData.text,
        reviewerName: reviewData.name,
        tonality,
        length,
        reviewType,
        formality: formality || 'sie'
      });

      if (result.error) {
        throw new Error(result.error);
      }

      console.log('[CLX Content] Response received:', result.response?.substring(0, 50) + '...');

      // Insert into textarea
      textarea.value = result.response;
      textarea.dispatchEvent(new Event('input', { bubbles: true }));
      textarea.dispatchEvent(new Event('change', { bubbles: true }));
      textarea.focus();

      // Show success
      messageEl.textContent = '✓ Antwort eingefügt!';
      messageEl.className = 'clx-message clx-success';

      // Close panel after delay
      setTimeout(() => panel.remove(), 2000);

    } catch (error) {
      console.error('[CLX Content] Generate error:', error);
      messageEl.textContent = '✗ ' + error.message;
      messageEl.className = 'clx-message clx-error';
    }

    generateBtn.disabled = false;
    generateBtn.innerHTML = `
      <svg viewBox="0 0 24 24" fill="currentColor" width="18" height="18">
        <path d="M12 2L14.39 8.26L21 9.27L16.5 13.97L17.77 20.72L12 17.27L6.23 20.72L7.5 13.97L3 9.27L9.61 8.26L12 2Z"/>
      </svg>
      Generieren
    `;
  }

  /**
   * Reset button state
   */
  function resetButton(btn) {
    btn.disabled = false;
    btn.innerHTML = `
      <svg viewBox="0 0 24 24" fill="currentColor" width="18" height="18">
        <path d="M12 2L14.39 8.26L21 9.27L16.5 13.97L17.77 20.72L12 17.27L6.23 20.72L7.5 13.97L3 9.27L9.61 8.26L12 2Z"/>
      </svg>
      <span>Erfassen</span>
    `;
  }

  /**
   * Show error message
   */
  function showError(btn, message) {
    const error = document.createElement('div');
    error.className = 'clx-error-toast';
    error.textContent = message;
    btn.parentElement.appendChild(error);
    setTimeout(() => error.remove(), 5000);
  }

  /**
   * Get review data from parent frame (when in iframe)
   */
  function getReviewDataFromParent() {
    return new Promise((resolve) => {
      const messageId = 'clx-' + Date.now();
      
      function handleResponse(event) {
        if (event.data.type === 'CLX_REVIEW_DATA_RESPONSE' && event.data.messageId === messageId) {
          window.removeEventListener('message', handleResponse);
          resolve(event.data.reviewData);
        }
      }
      
      window.addEventListener('message', handleResponse);
      
      // Request data from parent
      window.parent.postMessage({
        type: 'CLX_REVIEW_DATA_REQUEST',
        messageId: messageId
      }, '*');
      
      // Timeout after 2 seconds
      setTimeout(() => {
        window.removeEventListener('message', handleResponse);
        console.log('[CLX Content] Timeout - using default review data');
        resolve({ name: 'Kunde', text: '', stars: 5 });
      }, 2000);
    });
  }

  /**
   * Listen for review data requests from iframe (when on main page)
   */
  function setupFrameListener() {
    if (window === window.top) {
      window.addEventListener('message', (event) => {
        if (event.data.type === 'CLX_REVIEW_DATA_REQUEST') {
          console.log('[CLX Content] Parent received request for review data');
          
          // Find all review articles on the main page
          const articles = document.querySelectorAll('article.VaHEVc');
          console.log('[CLX Content] Found', articles.length, 'review articles on main page');
          
          let reviewData = { name: 'Kunde', text: '', stars: 5 };
          
          // Try to find the article that corresponds to the currently open reply dialog
          // Look for the article that has a visible textarea in the iframe
          if (articles.length > 0) {
            // Get the first article (usually the one with the open reply dialog)
            const firstArticle = articles[0];
            reviewData = extractReviewDataFromMainPage(firstArticle);
          }
          
          console.log('[CLX Content] Extracted review data:', reviewData);
          
          // Send data back to iframe
          event.source.postMessage({
            type: 'CLX_REVIEW_DATA_RESPONSE',
            messageId: event.data.messageId,
            reviewData: reviewData
          }, '*');
        }
      });
      console.log('[CLX Content] Frame listener setup (main page)');
    }
  }

  /**
   * Extract review data from main page (different structure than iframe)
   */
  function extractReviewDataFromMainPage(article) {
    const data = { name: 'Kunde', text: '', stars: 5 };
    
    console.log('[CLX Content] Extracting from main page article');
    console.log('[CLX Content] Article HTML preview:', article.innerHTML.substring(0, 200));

    // Strategy 1: Find reviewer name - multiple approaches
    let nameElement = article.querySelector('a.PskQHd.jEgW2b');
    if (!nameElement) nameElement = article.querySelector('a[jsname="xs1xe"]');
    if (!nameElement) nameElement = article.querySelector('.N0c6q.JhRJje a');
    if (!nameElement) nameElement = article.querySelector('.PskQHd');
    
    if (nameElement) {
      data.name = nameElement.textContent.trim();
      console.log('[CLX Content] ✓ Found name:', data.name);
    } else {
      console.log('[CLX Content] ✗ Name not found');
    }

    // Strategy 2: Find star rating - count icons or use aria-label
    const starContainer = article.querySelector('.DYizzd[aria-label]');
    if (starContainer) {
      const ariaLabel = starContainer.getAttribute('aria-label');
      const match = ariaLabel.match(/(\d+)\s+von\s+\d+/);
      if (match) {
        data.stars = parseInt(match[1]);
        console.log('[CLX Content] ✓ Found stars from aria-label:', data.stars);
      }
    }
    
    // Fallback: count star icons
    if (!data.stars || data.stars === 5) {
      const starIcons = article.querySelectorAll('.DYizzd i.lMAmUc, .DYizzd i.material-icons-extended');
      if (starIcons.length > 0 && starIcons.length <= 5) {
        data.stars = starIcons.length;
        console.log('[CLX Content] ✓ Found stars by counting:', data.stars);
      }
    }

    // Strategy 3: Find review text - the actual review content
    // Look for .gyKkFe.JhRJje.Fv38Af or .gyKkFe.Fv38Af
    const allTextDivs = article.querySelectorAll('.gyKkFe, .Fv38Af, div, span');
    
    for (const div of allTextDivs) {
      const text = div.textContent.trim();
      
      // Filter criteria: must be substantial review text
      const isSubstantial = text.length > 20;
      const isNotMetadata = !text.includes('Stern') && 
                           !text.includes('Vor ') && 
                           !text.includes('Rezension') &&
                           !text.includes('Fotos') &&
                           !text.includes('Local Guide') &&
                           !text.includes('Inhaber') &&
                           !text.includes('celox.io') &&
                           !text.includes('Neu') &&
                           !text.match(/^\d+\.\d+\.\d{4}$/) &&
                           !text.match(/^\d+\s*Rezension/) &&
                           !text.match(/^\d+\s*Fotos/);
      
      // Check if this div has the review text class
      const hasReviewClass = div.classList.contains('gyKkFe') || 
                            div.classList.contains('Fv38Af') ||
                            (div.classList.contains('gyKkFe') && div.classList.contains('JhRJje'));
      
      if (isSubstantial && isNotMetadata && hasReviewClass) {
        data.text = text;
        console.log('[CLX Content] ✓ Found review text:', text.substring(0, 80) + '...');
        break;
      }
    }
    
    if (!data.text) {
      console.log('[CLX Content] ✗ Review text not found');
    }

    console.log('[CLX Content] Final extracted data:', data);
    return data;
  }

  /**
   * Send message to background
   */
  function sendMessage(message) {
    return new Promise((resolve) => {
      chrome.runtime.sendMessage(message, (response) => {
        if (chrome.runtime.lastError) {
          console.error('[CLX Content] Message error:', chrome.runtime.lastError);
          resolve({ error: chrome.runtime.lastError.message });
        } else {
          resolve(response || { error: 'No response' });
        }
      });
    });
  }

  // Start
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

  // Re-scan periodically
  setInterval(scanForReviews, 3000);

})();
