/**
 * CLX GMaps AI Review - Background Service Worker
 * Combined file with all utilities to avoid importScripts issues
 */

console.log('[CLX Background] Service worker starting...');

// ============================================
// LOGGER MODULE
// ============================================
const CLXLogger = (function() {
  const LOG_STORAGE_KEY = 'clx_gmaps_logs';
  const MAX_LOGS = 1000;
  let logs = [];

  async function init() {
    try {
      if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.local) {
        const result = await chrome.storage.local.get(LOG_STORAGE_KEY);
        logs = result[LOG_STORAGE_KEY] || [];
        console.log('[CLX Logger] Initialized with', logs.length, 'logs');
      } else {
        logs = [];
        console.log('[CLX Logger] Initialized (no storage available)');
      }
    } catch (error) {
      logs = [];
      // Don't log init errors
    }
  }

  async function saveLogs() {
    try {
      if (logs.length > MAX_LOGS) logs = logs.slice(-MAX_LOGS);
      if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.local) {
        await chrome.storage.local.set({ [LOG_STORAGE_KEY]: logs });
      }
    } catch (e) {
      // Silently fail - storage might be unavailable
    }
  }

  function log(level, module, message, data = null) {
    const entry = { timestamp: new Date().toISOString(), level, module, message, data };
    logs.push(entry);
    console.log(`[CLX ${level}] [${module}] ${message}`, data || '');
    saveLogs();
    return entry;
  }

  init();

  return {
    init,
    debug: (m, msg, d) => log('DEBUG', m, msg, d),
    info: (m, msg, d) => log('INFO', m, msg, d),
    warn: (m, msg, d) => log('WARN', m, msg, d),
    error: (m, msg, d) => log('ERROR', m, msg, d),
    success: (m, msg, d) => log('SUCCESS', m, msg, d),
    getLogs: () => [...logs],
    getLogCount: () => logs.length,
    clearLogs: async () => { logs = []; await saveLogs(); }
  };
})();

// ============================================
// LICENSE MODULE
// ============================================
// Shared LICENSE_SERVER constant (used by both CLXLicense and CLXAPI)
const LICENSE_SERVER = 'https://clx-gmaps-ai-review.celox.io';

const CLXLicense = (function() {
  const STORAGE_KEY = 'clx_gmaps_license';
  const FREE_GENERATIONS = 3;
  let licenseData = null;

  function generateUUID() {
    return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
      const r = Math.random() * 16 | 0;
      return (c === 'x' ? r : (r & 0x3 | 0x8)).toString(16);
    });
  }

  /**
   * Generate browser fingerprint for user identification
   * Works in Service Worker context (no DOM access)
   */
  async function generateBrowserFingerprint() {
    const components = [];
    
    // User Agent (available in Service Worker)
    if (typeof navigator !== 'undefined' && navigator.userAgent) {
      components.push(navigator.userAgent);
    }
    
    // Timezone
    try {
      components.push(Intl.DateTimeFormat().resolvedOptions().timeZone);
    } catch (e) {}
    
    // Language
    if (typeof navigator !== 'undefined') {
      components.push(navigator.language || navigator.userLanguage || '');
      if (navigator.languages && navigator.languages.length > 0) {
        components.push(navigator.languages.join(','));
      }
    }
    
    // Platform
    if (typeof navigator !== 'undefined' && navigator.platform) {
      components.push(navigator.platform);
    }
    
    // Hardware concurrency (CPU cores)
    if (typeof navigator !== 'undefined' && navigator.hardwareConcurrency) {
      components.push(`cores:${navigator.hardwareConcurrency}`);
    }
    
    // Device memory (if available)
    if (typeof navigator !== 'undefined' && navigator.deviceMemory) {
      components.push(`mem:${navigator.deviceMemory}`);
    }
    
    // Chrome Extension ID (if available)
    try {
      if (typeof chrome !== 'undefined' && chrome.runtime && chrome.runtime.id) {
        components.push(`ext:${chrome.runtime.id}`);
      }
    } catch (e) {}
    
    // Get screen info from storage (set by popup/content script)
    try {
      const stored = await chrome.storage.local.get('clx_fingerprint_screen');
      if (stored.clx_fingerprint_screen) {
        components.push(stored.clx_fingerprint_screen);
      }
    } catch (e) {}
    
    // Combine all components
    const fingerprintString = components.join('|');
    
    // Create a simple hash (not cryptographically secure, but sufficient for fingerprinting)
    let hash = 0;
    for (let i = 0; i < fingerprintString.length; i++) {
      const char = fingerprintString.charCodeAt(i);
      hash = ((hash << 5) - hash) + char;
      hash = hash & hash; // Convert to 32-bit integer
    }
    
    return Math.abs(hash).toString(16);
  }

  function getGenerationsRemaining() {
    if (!licenseData) return FREE_GENERATIONS;
    const totalAvailable = FREE_GENERATIONS + (licenseData.purchasedGenerations || 0);
    const used = licenseData.totalGenerations || 0;
    return Math.max(0, totalAvailable - used);
  }

  async function saveLicenseData() {
    try {
      await chrome.storage.local.set({ [STORAGE_KEY]: licenseData });
    } catch (e) {
      console.error('[CLX License] Save error:', e);
    }
  }

  async function registerUUID(uuid, installedAt) {
    try {
      // Generate browser fingerprint for user identification
      const fingerprint = await generateBrowserFingerprint();
      
      const response = await fetch(`${LICENSE_SERVER}/license/register`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ 
          uuid, 
          installedAt,
          fingerprint 
        })
      });
      
      if (!response.ok) {
        console.warn('[CLX License] Registration failed, but continuing');
        return { allowTrial: true }; // Default: allow trial if server fails
      }
      
      const result = await response.json();
      return result;
    } catch (error) {
      console.error('[CLX License] Registration error:', error);
      // If server is unreachable, allow trial (graceful degradation)
      return { allowTrial: true };
    }
  }

  async function init() {
    try {
      const result = await chrome.storage.local.get(STORAGE_KEY);
      licenseData = result[STORAGE_KEY];

      if (!licenseData) {
        // New installation - generate UUID
        const uuid = generateUUID();
        const installedAt = new Date().toISOString();
        
        // Register UUID on server
        const registration = await registerUUID(uuid, installedAt);
        
        licenseData = {
          uuid: uuid,
          installedAt: installedAt,
          licenseStatus: 'free',
          lastCheck: null,
          totalGenerations: 0,
          purchasedGenerations: 0
        };
        await saveLicenseData();
        console.log('[CLX License] New installation - 3 free generations, UUID:', uuid);
      } else {
        console.log('[CLX License] Loaded, UUID:', licenseData.uuid);
      }
      return licenseData;
    } catch (error) {
      console.error('[CLX License] Init error:', error);
      // On error, create minimal license data
      const uuid = generateUUID();
      licenseData = {
        uuid: uuid,
        installedAt: new Date().toISOString(),
        licenseStatus: 'free',
        lastCheck: null,
        totalGenerations: 0,
        purchasedGenerations: 0
      };
      return licenseData;
    }
  }


  async function checkLicense() {
    try {
      const response = await fetch(`${LICENSE_SERVER}/license/check`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ uuid: licenseData.uuid })
      });
      const result = await response.json();
      
      licenseData.lastCheck = new Date().toISOString();
      
      if (result.valid) {
        licenseData.licenseStatus = 'active';
        licenseData.totalGenerations = result.total_generations || 0;
        licenseData.purchasedGenerations = result.purchased_generations || 0;
        licenseData.freeGenerationsRemaining = result.free_generations_remaining || 0;
        licenseData.totalGenerationsRemaining = result.total_generations_remaining || 0;
        await saveLicenseData();
        
        console.log('[CLX License] License verified', result.total_generations_remaining, 'generations remaining');
        
        return {
          valid: true,
          status: 'active',
          totalGenerationsRemaining: result.total_generations_remaining,
          freeGenerationsRemaining: result.free_generations_remaining,
          purchasedGenerations: result.purchased_generations,
          totalGenerations: result.total_generations,
          user: result.user
        };
      } else {
        licenseData.licenseStatus = result.reason === 'no_generations' ? 'no_generations' : 'expired';
        licenseData.totalGenerations = result.total_generations || 0;
        licenseData.purchasedGenerations = result.purchased_generations || 0;
        await saveLicenseData();
        
        console.log('[CLX License] License not valid', result.reason);
        
        return {
          valid: false,
          status: result.reason === 'no_generations' ? 'no_generations' : 'expired',
          reason: result.reason,
          message: result.message,
          totalGenerations: result.total_generations,
          purchasedGenerations: result.purchased_generations,
          needsPayment: result.reason === 'no_generations',
          user: result.user
        };
      }
    } catch (error) {
      console.error('[CLX License] Check error:', error);
      
      // If server is unreachable, allow usage if we have remaining generations locally
      const remaining = getGenerationsRemaining();
      if (remaining > 0) {
        return {
          valid: true,
          status: 'active',
          totalGenerationsRemaining: remaining,
          offline: true
        };
      }
      
      return { valid: false, status: 'error', message: 'Server nicht erreichbar. Bitte Internetverbindung prüfen.' };
    }
  }

  function getStatusString() {
    if (!licenseData) return 'Nicht initialisiert';
    
    const remaining = licenseData.totalGenerationsRemaining !== undefined 
      ? licenseData.totalGenerationsRemaining 
      : getGenerationsRemaining();
    
    if (licenseData.licenseStatus === 'active' && remaining > 0) {
      const purchased = licenseData.purchasedGenerations || 0;
      if (purchased > 0) {
        return `${remaining} Generierung${remaining !== 1 ? 'en' : ''} verfügbar`;
      }
      return `${remaining}/3 kostenlose Generierungen`;
    }
    
    if (licenseData.licenseStatus === 'no_generations') {
      return 'Alle Generierungen aufgebraucht';
    }
    
    return 'Nicht verfügbar';
  }

  function getStatusType() {
    if (!licenseData) return 'expired';
    
    const remaining = licenseData.totalGenerationsRemaining !== undefined 
      ? licenseData.totalGenerationsRemaining 
      : getGenerationsRemaining();
    
    if (licenseData.licenseStatus === 'active' && remaining > 0) {
      return 'active';
    }
    
    if (licenseData.licenseStatus === 'no_generations') {
      return 'expired';
    }
    
    return 'expired';
  }

  async function canUse() {
    const result = await checkLicense();
    return result.valid;
  }

  // Store init promise
  let initPromise = null;

  // Public API
  return {
    init: async () => {
      if (!initPromise) {
        initPromise = init();
      }
      return await initPromise;
    },
    getLicenseData: async () => {
      // Ensure initialization
      if (!licenseData) {
        if (!initPromise) {
          initPromise = init();
        }
        await initPromise;
      }
      // Return copy of license data, ensure UUID exists
      if (licenseData && licenseData.uuid) {
        return { ...licenseData };
      }
      // If no UUID, force re-init
      await init();
      return licenseData ? { ...licenseData } : null;
    },
    getGenerationsRemaining,
    checkLicense: async () => {
      if (!initPromise) {
        initPromise = init();
      }
      await initPromise;
      return await checkLicense();
    },
    getStatusString,
    getStatusType,
    canUse: async () => {
      if (!initPromise) {
        initPromise = init();
      }
      await initPromise;
      const result = await checkLicense();
      return result.valid;
    }
  };
})();

// ============================================
// API MODULE
// ============================================
const CLXAPI = (function() {
  const STORAGE_KEY = 'clx_gmaps_settings';
  let settings = null;

  async function init() {
    try {
      const result = await chrome.storage.local.get(STORAGE_KEY);
      const rawSettings = result[STORAGE_KEY] || {
        openaiKey: '',
        anthropicKey: '',
        selectedAI: 'openai',
        defaultLength: 'normal',
        formality: 'sie'
      };
      
      // Clean and convert to strings explicitly
      settings = {
        openaiKey: String(rawSettings.openaiKey || '').trim(),
        anthropicKey: String(rawSettings.anthropicKey || '').trim(),
        selectedAI: rawSettings.selectedAI || 'openai',
        defaultLength: rawSettings.defaultLength || 'normal',
        formality: rawSettings.formality || 'sie',
        lastTonality: rawSettings.lastTonality || { positive: 'herzlich', neutral: 'professionell', negative: 'empathisch' },
        lastLength: rawSettings.lastLength || 'normal'
      };
      
      console.log('[CLX API] Settings loaded:', { 
        selectedAI: settings.selectedAI,
        hasOpenAI: !!settings.openaiKey,
        hasAnthropic: !!settings.anthropicKey,
        openaiKeyLength: settings.openaiKey.length,
        anthropicKeyLength: settings.anthropicKey.length
      });
      return settings;
    } catch (error) {
      console.error('[CLX API] Init error:', error);
      settings = { openaiKey: '', anthropicKey: '', selectedAI: 'openai', defaultLength: 'normal' };
      return settings;
    }
  }

  async function saveSettings(newSettings) {
    if (!settings) await init();
    
    // Merge new settings with existing ones (no API keys needed anymore)
    const cleanedSettings = {
      openaiKey: String(settings.openaiKey || '').trim(), // Keep existing (empty now)
      anthropicKey: String(settings.anthropicKey || '').trim(), // Keep existing (empty now)
      selectedAI: newSettings.selectedAI || settings.selectedAI || 'openai',
      defaultLength: newSettings.defaultLength || settings.defaultLength || 'normal',
      formality: newSettings.formality || settings.formality || 'sie',
      lastTonality: newSettings.lastTonality || settings.lastTonality || { positive: 'herzlich', neutral: 'professionell', negative: 'empathisch' },
      lastLength: newSettings.lastLength || settings.lastLength || 'normal',
      screenInfo: newSettings.screenInfo || settings.screenInfo || {}
    };
    
    settings = cleanedSettings;
    
    try {
      await chrome.storage.local.set({ [STORAGE_KEY]: settings });
      console.log('[CLX API] Settings saved', {
        selectedAI: settings.selectedAI,
        formality: settings.formality,
        defaultLength: settings.defaultLength
      });
      return true;
    } catch (error) {
      console.error('[CLX API] Save error:', error);
      return false;
    }
  }

  async function getSettings() {
    if (!settings) await init();
    return settings ? { ...settings } : { 
      openaiKey: '', 
      anthropicKey: '', 
      selectedAI: 'openai', 
      defaultLength: 'normal', 
      formality: 'sie',
      lastTonality: { positive: 'herzlich', neutral: 'professionell', negative: 'empathisch' },
      lastLength: 'normal'
    };
  }

  async function classifyReview(reviewText, reviewerName, starRating = null) {
    if (starRating !== null) {
      let type = starRating >= 4 ? 'positive' : starRating === 3 ? 'neutral' : 'negative';
      return { type, confidence: 1.0 };
    }
    // Default to neutral if no stars
    return { type: 'neutral', confidence: 0.5 };
  }

  function getCleanApiKey(key) {
    if (!key) return '';
    // Convert to string and trim
    let cleaned = String(key).trim();
    // Remove any invisible characters
    cleaned = cleaned.replace(/[\u200B-\u200D\uFEFF]/g, '');
    return cleaned;
  }

  async function generateResponse(reviewText, reviewerName, tonality, length, reviewType, formality = 'sie') {
    if (!settings) await init();

    // Get UUID for server request
    const licenseData = await CLXLicense.getLicenseData();
    if (!licenseData || !licenseData.uuid) {
      throw new Error('UUID nicht gefunden. Bitte Plugin neu laden.');
    }

    // Use server endpoint for generation (server handles API keys)
    const provider = settings.selectedAI || 'openai';
    
    try {
      const response = await fetch(`${LICENSE_SERVER}/api/generate`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          uuid: licenseData.uuid,
          reviewText,
          reviewerName,
          tonality,
          length,
          reviewType,
          formality,
          provider
        })
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Server-Fehler bei der Generierung');
      }

      const data = await response.json();
      console.log(`[CLX API] Response generated via server, tokens used: ${data.tokens_used || 0}`);
      return data.response;
    } catch (error) {
      console.error('[CLX API] Server generation error:', error);
      throw error;
    }
  }

  async function testConnection(aiType, openaiKey, anthropicKey) {
    // API Keys werden jetzt zentral auf dem Server verwaltet
    // Diese Funktion wird nicht mehr benötigt, aber für Kompatibilität behalten
    return { success: true, message: 'API Keys werden zentral auf dem Server verwaltet' };
  }

  init();

  return { init, saveSettings, getSettings, classifyReview, generateResponse, testConnection };
})();

// ============================================
// INITIALIZE MODULES
// ============================================
(async () => {
  try {
    await CLXLogger.init();
    await CLXLicense.init();
    await CLXAPI.init();
    console.log('[CLX Background] All modules initialized');
  } catch (error) {
    console.error('[CLX Background] Initialization error:', error);
  }
})();

// Ensure service worker stays alive
chrome.runtime.onInstalled.addListener(() => {
  console.log('[CLX Background] Extension installed/updated');
});

chrome.runtime.onStartup.addListener(() => {
  console.log('[CLX Background] Extension startup');
});

// ============================================
// MESSAGE HANDLER
// ============================================
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  console.log('[CLX Background] Message received:', request.action);
  
  // Handle message asynchronously
  handleMessage(request)
    .then(response => {
      console.log('[CLX Background] Response:', response);
      try {
        sendResponse(response);
      } catch (error) {
        console.error('[CLX Background] Error sending response:', error);
      }
    })
    .catch(error => {
      console.error('[CLX Background] Error handling message:', error);
      try {
        sendResponse({ error: error.message });
      } catch (sendError) {
        console.error('[CLX Background] Error sending error response:', sendError);
      }
    });
  
  return true; // Keep message channel open for async response
});

async function handleMessage(request) {
  switch (request.action) {
    case 'getUUID':
      // Get UUID from license data
      await CLXLicense.init();
      const uuidData = await CLXLicense.getLicenseData();
      return { uuid: uuidData?.uuid || null };

    case 'checkLicense':
      return await CLXLicense.checkLicense();

    case 'getLicenseData':
      // Ensure license is initialized
      await CLXLicense.init();
      
      // FIRST check license to get fresh data from server
      const checkResult = await CLXLicense.checkLicense();
      
      // THEN get the updated license data
      const licData = await CLXLicense.getLicenseData();
      
      if (!licData) {
        return { error: 'License data not available' };
      }
      
      const result = {
        ...licData,
        statusString: CLXLicense.getStatusString(),
        statusType: CLXLicense.getStatusType()
      };
      
      // Add generation info from checkResult
      if (checkResult) {
        result.needsPayment = checkResult.needsPayment || false;
        result.totalGenerations = checkResult.totalGenerations;
        result.purchasedGenerations = checkResult.purchasedGenerations;
        result.totalGenerationsRemaining = checkResult.totalGenerationsRemaining;
        result.user = checkResult.user;
      }
      
      return result;

    case 'classifyReview':
      // Classify review via server (counts as generation)
      const licenseData = await CLXLicense.getLicenseData();
      if (!licenseData || !licenseData.uuid) {
        return { error: 'UUID nicht verfügbar' };
      }
      
      try {
        const response = await fetch(`${LICENSE_SERVER}/api/classify`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json'
          },
          body: JSON.stringify({
            uuid: licenseData.uuid,
            reviewText: request.reviewText,
            reviewerName: request.reviewerName,
            starRating: request.starRating
          })
        });
        
        if (!response.ok) {
          const error = await response.json();
          throw new Error(error.error || 'Server-Fehler bei der Klassifikation');
        }
        
        const result = await response.json();
        console.log('[CLX Background] Classification result:', result);
        return result;
      } catch (error) {
        console.error('[CLX Background] Classification error:', error);
        // Fallback to local classification if server fails
        return await CLXAPI.classifyReview(request.reviewText, request.reviewerName, request.starRating);
      }

    case 'generateResponse':
      const canUse = await CLXLicense.canUse();
      if (!canUse) {
        const licenseData = await CLXLicense.getLicenseData();
        const checkResult = await CLXLicense.checkLicense();
        if (checkResult.needsPayment) {
          return { 
            error: 'Alle Generierungen aufgebraucht. Bitte kaufen Sie weitere Generierungen.',
            needsPayment: true,
            totalGenerations: checkResult.totalGenerations,
            purchasedGenerations: checkResult.purchasedGenerations
          };
        }
        return { error: 'Lizenz abgelaufen oder ungültig' };
      }
      
      // Save last tonality and length before generating
      if (request.reviewType && request.tonality && request.length) {
        await CLXAPI.saveSettings({
          lastTonality: {
            ...((await CLXAPI.getSettings()).lastTonality || { positive: 'herzlich', neutral: 'professionell', negative: 'empathisch' }),
            [request.reviewType]: request.tonality
          },
          lastLength: request.length
        });
      }
      
      const response = await CLXAPI.generateResponse(
        request.reviewText, request.reviewerName, request.tonality, request.length, request.reviewType, request.formality
      );
      return { response };

    case 'getSettings':
      return await CLXAPI.getSettings();

    case 'saveSettings':
      await CLXAPI.saveSettings(request.settings);
      return { success: true };

        case 'testConnection':
          return await CLXAPI.testConnection(request.aiType, request.openaiKey, request.anthropicKey);

    case 'getLogs':
      return { logs: CLXLogger.getLogs(), count: CLXLogger.getLogCount() };

    case 'exportLogs':
      return { logs: CLXLogger.getLogs(), count: CLXLogger.getLogCount() };

    case 'clearLogs':
      await CLXLogger.clearLogs();
      return { success: true };

    default:
      return { error: 'Unknown action' };
  }
}

console.log('[CLX Background] Service worker ready');
